/*
 * midi.h
 *
 *  Created on: Feb 3, 2015
 *      Author: Austin Riffle
 */

#ifndef MIDI_H_
#define MIDI_H_

#include "globalDefs.h"
#include "UartInterface.h"
#include "MidiDefinitions.h"
#include "MidiMessage.h"

using uart::UartInterface;

namespace midi
{
// MidiInterface class
class MidiInterface
{
private:
	struct Settings
	{
	    /*! Running status enables short messages when sending multiple values
	     of the same type and channel.\n
	     Set to 0 if you have troubles controlling your hardware.
	     */
	    static const bool UseRunningStatus = true;

	    /* NoteOn with 0 velocity should be handled as NoteOf.\n
	     Set to 1 to get NoteOff events when receiving null-velocity NoteOn messages.\n
	     Set to 0 to get NoteOn  events when receiving null-velocity NoteOn messages.
	     */
	    static const bool HandleNullVelocityNoteOnAsNoteOff = true;

	    // Setting this to 1 will make MIDI.read parse only one byte of data for each
	    // call when data is available. This can speed up your application if receiving
	    // a lot of traffic, but might induce MIDI Thru and treatment latency.
	    static const bool Use1ByteParsing = true;

	    /*! Override the default MIDI baudrate to transmit over USB serial, to
	    a decoding program such as Hairless MIDI (set baudrate to 115200)\n
	    http://projectgus.github.io/hairless-midiserial/
	    */
	    static const long BaudRate = 31250;

	    /*! Maximum size of SysEx receivable. Decrease to save RAM if you don't expect
	    to receive SysEx, or adjust accordingly.
	    */
	    static const unsigned SysExMaxSize = 128;
	};

public:
    MidiInterface();
    ~MidiInterface();

public:
    void begin(Channel inChannel = 1);

    // -------------------------------------------------------------------------
    // MIDI Output

public:
    void sendNoteOn(DataByte inNoteNumber, DataByte inVelocity, Channel inChannel);
    void sendNoteOff(DataByte inNoteNumber, DataByte inVelocity, Channel inChannel);
    void sendProgramChange(DataByte inProgramNumber, Channel inChannel);
    void sendControlChange(DataByte inControlNumber, DataByte inControlValue, Channel inChannel);
    void sendPitchBend(int inPitchValue,    Channel inChannel);
    void sendPitchBend(double inPitchValue, Channel inChannel);
    void sendPolyPressure(DataByte inNoteNumber, DataByte inPressure, Channel inChannel);
    void sendAfterTouch(DataByte inPressure, Channel inChannel);
    void sendSysEx(unsigned inLength, const uint8_t* inArray,
    			   bool inArrayContainsBoundaries = false);
    void sendTimeCodeQuarterFrame(DataByte inTypeNibble, DataByte inValuesNibble);
    void sendTimeCodeQuarterFrame(DataByte inData);
    void sendSongPosition(unsigned inBeats);
    void sendSongSelect(DataByte inSongNumber);
    void sendTuneRequest();
    void sendRealTime(MidiType inType);

public:
    void send(MidiType inType, DataByte inData1, DataByte inData2, Channel inChannel);

    // -------------------------------------------------------------------------
    // MIDI Input

public:
    bool read();
    bool read(Channel inChannel);

public:
    MidiType getType() const;
    Channel  getChannel() const;
    DataByte getData1() const;
    DataByte getData2() const;
    const uint8_t* getSysExArray() const;
    unsigned getSysExArrayLength() const;
    bool check() const;

public:
    Channel getInputChannel() const;
    void setInputChannel(Channel inChannel);

public:
    static MidiType getTypeFromStatusByte(uint8_t inStatus);
    static Channel getChannelFromStatusByte(uint8_t inStatus);
	static bool isChannelMessage(MidiType inType);


    // -------------------------------------------------------------------------
    // Input Callbacks

public:
    void setHandleNoteOff(void (*fptr)(uint8_t channel, uint8_t note, uint8_t velocity));
    void setHandleNoteOn(void (*fptr)(uint8_t channel, uint8_t note, uint8_t velocity));
    void setHandleAfterTouchPoly(void (*fptr)(uint8_t channel, uint8_t note, uint8_t pressure));
    void setHandleControlChange(void (*fptr)(uint8_t channel, uint8_t number, uint8_t value));
    void setHandleProgramChange(void (*fptr)(uint8_t channel, uint8_t number));
    void setHandleAfterTouchChannel(void (*fptr)(uint8_t channel, uint8_t pressure));
    void setHandlePitchBend(void (*fptr)(uint8_t channel, int bend));
    void setHandleSystemExclusive(void (*fptr)(uint8_t * array, unsigned size));
    void setHandleTimeCodeQuarterFrame(void (*fptr)(uint8_t data));
    void setHandleSongPosition(void (*fptr)(unsigned beats));
    void setHandleSongSelect(void (*fptr)(uint8_t songnumber));
    void setHandleTuneRequest(void (*fptr)(void));
    void setHandleClock(void (*fptr)(void));
    void setHandleStart(void (*fptr)(void));
    void setHandleContinue(void (*fptr)(void));
    void setHandleStop(void (*fptr)(void));
    void setHandleActiveSensing(void (*fptr)(void));
    void setHandleSystemReset(void (*fptr)(void));

    void disconnectCallbackFromType(MidiType inType);

private:
    void launchCallback();

    void (*mNoteOffCallback)(uint8_t channel, uint8_t note, uint8_t velocity);
    void (*mNoteOnCallback)(uint8_t channel, uint8_t note, uint8_t velocity);
    void (*mAfterTouchPolyCallback)(uint8_t channel, uint8_t note, uint8_t velocity);
    void (*mControlChangeCallback)(uint8_t channel, uint8_t, uint8_t);
    void (*mProgramChangeCallback)(uint8_t channel, uint8_t);
    void (*mAfterTouchChannelCallback)(uint8_t channel, uint8_t);
    void (*mPitchBendCallback)(uint8_t channel, int);
    void (*mSystemExclusiveCallback)(uint8_t * array, unsigned size);
    void (*mTimeCodeQuarterFrameCallback)(uint8_t data);
    void (*mSongPositionCallback)(unsigned beats);
    void (*mSongSelectCallback)(uint8_t songnumber);
    void (*mTuneRequestCallback)(void);
    void (*mClockCallback)(void);
    void (*mStartCallback)(void);
    void (*mContinueCallback)(void);
    void (*mStopCallback)(void);
    void (*mActiveSensingCallback)(void);
    void (*mSystemResetCallback)(void);

    // -------------------------------------------------------------------------
    // MIDI Soft Thru

public:
    MidiFilterMode getFilterMode() const;
    bool getThruState() const;

    void turnThruOn(MidiFilterMode inThruFilterMode = Full);
    void turnThruOff();
    void setThruFilterMode(MidiFilterMode inThruFilterMode);

private:
    void thruFilter(uint8_t inChannel);

private:
    bool parse();
    void handleNullVelocityNoteOnAsNoteOff();
    bool inputFilter(Channel inChannel);
    void resetInput();

private:
    bool            mThruActivated  : 1;
    MidiFilterMode  mThruFilterMode : 7;

private:
    StatusByte  mRunningStatus_RX;
    StatusByte  mRunningStatus_TX;
    Channel     mInputChannel;
    uint8_t     mPendingMessage[3];
    unsigned    mPendingMessageExpectedLenght;
    unsigned    mPendingMessageIndex;
    MidiMessage mMessage;

private:
    StatusByte getStatus(MidiType inType, Channel inChannel) const;

private:
    UartInterface mSerial;
};

}
#endif /* MIDI_H_ */
